CREATE OR REPLACE PACKAGE TPX_ATTENDANCE AS
/*==============================================================================
    Object Name  : TPX_ATTENDANCE
    Object Type  : PACKAGE
    Description  : 출퇴근 관리 패키지
==============================================================================*/

    -- 출퇴근 등록
    PROCEDURE INS_ATTENDANCE (
        iEmpKey     IN VARCHAR2,
        iWorkDate   IN DATE,
        iStatus     IN VARCHAR2
    );
    
    -- 결근 일수 조회
    FUNCTION GET_ABSENT_DAYS (
        iEmpKey     IN VARCHAR2,
        iStartDate  IN DATE,
        iEndDate    IN DATE
    ) RETURN NUMBER;

END TPX_ATTENDANCE;
/

CREATE OR REPLACE PACKAGE BODY TPX_ATTENDANCE AS
/*==============================================================================
    Object Name  : TPX_ATTENDANCE
    Object Type  : PACKAGE
    Description  : 출퇴근 관리 패키지
==============================================================================*/

    -- 출퇴근 등록
    PROCEDURE INS_ATTENDANCE (
        iEmpKey     IN VARCHAR2,
        iWorkDate   IN DATE,
        iStatus     IN VARCHAR2
    ) IS
        vExists BOOLEAN;
    BEGIN
        BEGIN
            -- 1. 직원 존재 확인
            vExists := TPX_EMPLOYEE.EXISTS_EMPLOYEE(iEmpKey);
            IF NOT vExists THEN
                RAISE_APPLICATION_ERROR(-20000, '직원이 존재하지 않습니다.');
            END IF;
            
            -- 2. 출퇴근 등록
            INSERT INTO TPJ_ATTENDANCE (
                ATT_KEY, EMP_KEY, WORK_DATE, STATUS
            ) VALUES (
                SYS_GUID(),
                iEmpKey, 
                iWorkDate, 
                iStatus
            );
            
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in INS_ATTENDANCE: ' || SQLERRM);
        END;
    END INS_ATTENDANCE;

    -- 결근 일수 조회
    FUNCTION GET_ABSENT_DAYS (
        iEmpKey     IN VARCHAR2,
        iStartDate  IN DATE,
        iEndDate    IN DATE
    ) RETURN NUMBER IS
        vCount NUMBER;
        vExists BOOLEAN;
    BEGIN
        BEGIN
            -- 1. 직원 존재 확인
            vExists := TPX_EMPLOYEE.EXISTS_EMPLOYEE(iEmpKey);
            IF NOT vExists THEN
                RAISE_APPLICATION_ERROR(-20000, '직원이 존재하지 않습니다.');
            END IF;
            
            -- 2. 결근 일수 조회
            SELECT COUNT(*)
            INTO vCount
            FROM TPJ_ATTENDANCE
            WHERE EMP_KEY = iEmpKey
              AND WORK_DATE BETWEEN iStartDate AND iEndDate
              AND STATUS = 'AB';
              
            RETURN vCount;
            
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in GET_ABSENT_DAYS: ' || SQLERRM);
        END;
    END GET_ABSENT_DAYS;

END TPX_ATTENDANCE;
/