CREATE OR REPLACE PACKAGE TPX_SALARY AS
/*==============================================================================
    Object Name  : TPX_SALARY
    Object Type  : PACKAGE
    Description  : 급여 관리 패키지
==============================================================================*/


    -- 초기 급여 등록
    PROCEDURE INS_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE,
        iBaseAmount IN NUMBER
    );
    

    -- 결석에 따른 급여 수정
    PROCEDURE UPD_SALARY_BY_ABSENT (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    );
    

    -- 급여 조회
    FUNCTION GET_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    ) RETURN NUMBER;


    -- 급여 정보 존재 여부 확인 함수 추가
    FUNCTION EXISTS_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    ) RETURN BOOLEAN;


END TPX_SALARY;
/

CREATE OR REPLACE PACKAGE BODY TPX_SALARY AS
/*==============================================================================
    Object Name  : TPX_SALARY
    Object Type  : PACKAGE BODY
    Description  : 급여 관리 패키지
==============================================================================*/
    
    -- 급여 정보 존재 여부 확인
    FUNCTION EXISTS_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    ) RETURN BOOLEAN IS
        vCount NUMBER;
    BEGIN
        SELECT COUNT(*)
        INTO vCount
        FROM TPJ_SALARY
        WHERE EMP_KEY = iEmpKey
          AND PAY_DATE = TRUNC(iPayDate);
          
        RETURN vCount > 0;
    END EXISTS_SALARY;


    -- 결근당 차감액 계산 (Private)
    FUNCTION p_CALCULATE_DEDUCTION (
        iEmpKey     IN VARCHAR2,
        iAbsentDays IN NUMBER
    ) RETURN NUMBER IS
        vDeduction NUMBER;
        vDeductionPerDay NUMBER;
        vRegularYn CHAR(1);
    BEGIN
        BEGIN
            -- 정직원 여부 확인
            vRegularYn := TPX_EMPLOYEE.IS_REGULAR_EMPLOYEE(iEmpKey);
            
            -- 정직원 여부에 따른 일일 차감액 설정
            IF vRegularYn = 'Y' THEN
                vDeductionPerDay := 100000;  -- 정직원: 결근당 10만원 차감
            ELSE
                vDeductionPerDay := 200000;  -- 비정규직: 결근당 20만원 차감
            END IF;
            
            vDeduction := iAbsentDays * vDeductionPerDay;
            RETURN vDeduction;
        
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in p_CALCULATE_DEDUCTION: ' || SQLERRM);
        END;
    END p_CALCULATE_DEDUCTION;

    -- 급여 조회
    FUNCTION GET_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    ) RETURN NUMBER IS
        vAmount NUMBER;
    BEGIN
        BEGIN
            SELECT AMOUNT
            INTO vAmount
            FROM TPJ_SALARY
            WHERE EMP_KEY = iEmpKey
              AND PAY_DATE = TRUNC(iPayDate);
              
            RETURN vAmount;
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in GET_SALARY: ' || SQLERRM);
        END;
    END GET_SALARY;

    -- 초기 급여 등록
    PROCEDURE INS_SALARY (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE,
        iBaseAmount IN NUMBER
    ) IS
        vExists BOOLEAN;
    BEGIN
        BEGIN
            -- 1. 직원 존재 확인
            vExists := TPX_EMPLOYEE.EXISTS_EMPLOYEE(iEmpKey);
            IF NOT vExists THEN
                RAISE_APPLICATION_ERROR(-20000, '직원이 존재하지 않습니다.');
            END IF;
            
            -- 2. 급여 등록
            INSERT INTO TPJ_SALARY (
                SAL_KEY, EMP_KEY, PAY_DATE, AMOUNT
            ) VALUES (
                SYS_GUID(),
                iEmpKey, 
                TRUNC(iPayDate), 
                iBaseAmount
            );
            
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in INS_SALARY: ' || SQLERRM);
        END;
    END INS_SALARY;

    -- 결석에 따른 급여 수정
    PROCEDURE UPD_SALARY_BY_ABSENT (
        iEmpKey     IN VARCHAR2,
        iPayDate    IN DATE
    ) IS
        vBaseAmount NUMBER;
        vAbsentDays NUMBER;
        vFinalAmount NUMBER;
        vExists BOOLEAN;
    BEGIN
        BEGIN

            -- 1. 직원 존재 확인
            vExists := TPX_EMPLOYEE.EXISTS_EMPLOYEE(iEmpKey);
            IF NOT vExists THEN
                RAISE_APPLICATION_ERROR(-20000, '직원이 존재하지 않습니다.');
            END IF;

            -- 2. 기존 급여 조회
            vBaseAmount := GET_SALARY(iEmpKey, iPayDate);
            
            -- 3. 결근 일수 조회
            vAbsentDays := TPX_ATTENDANCE.GET_ABSENT_DAYS(
                iEmpKey,
                TRUNC(iPayDate, 'MM'),
                LAST_DAY(iPayDate)
            );
            
            -- 4. 최종 급여 계산
            vFinalAmount := vBaseAmount - p_CALCULATE_DEDUCTION(iEmpKey, vAbsentDays);
            
            -- 5. 급여 수정
            UPDATE TPJ_SALARY
            SET AMOUNT = vFinalAmount
            WHERE EMP_KEY = iEmpKey
              AND PAY_DATE = TRUNC(iPayDate);
              
        EXCEPTION
            WHEN OTHERS THEN
                RAISE_APPLICATION_ERROR(-20000, 'Error in UPD_SALARY_BY_ABSENT: ' || SQLERRM);
        END;
    END UPD_SALARY_BY_ABSENT;

END TPX_SALARY;
/